/*
 * The Butterfly XML Editor
 * http://www.butterflyxml.org
 * 
 * Copyright (C) 2004  Jules White
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Original Author: Jules White
 * Contributor(s):
 */
package butterfly.actions.contenthandlers;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;
import java.util.StringTokenizer;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.log4j.Logger;

import butterfly.actions.contenthandlers.interfaces.IContentType;

/**
 * @author jules
 *
 * To change this generated comment edit the template variable "typecomment":
 * Window>Preferences>Java>Templates.
 * To enable and disable the creation of type comments go to
 * Window>Preferences>Java>Code Generation.
 */
public class ContentType implements IContentType {
	private static final Logger logger_ = Logger.getLogger(ContentType.class);
	
	public final static String ALLOWED_VALUES_PREFIX = "allowed-values-for-";
	public final static String DESCRIPTION_PREFIX = "description-for-";
	public final static String DESCRIPTION_ATTRIBUTE = "description";
	public final static String PROPERTY_NAME_ATTRIBUTE = "name";
	
	private String displayName_;
	private String name_;
	private String[] fileExtensions_;
	private String description_;
	private Properties defaultProperties_ = new Properties();
	private IContentType parentContentType_;
	private static Hashtable contentTypes_ = new Hashtable(5);

	private ContentType(
		IContentType parent,
		String name,
		String displayn,
		String fileexts,		
		String desc) {
		name_ = name;
		displayName_ = displayn;
		parentContentType_ = parent;
		description_ = desc;
		fileExtensions_ = parseFileExtensionString(fileexts);
	}

	private String[] parseFileExtensionString(String fexts) {
		StringTokenizer tk = new StringTokenizer(fexts, ",");
		String[] exts = new String[tk.countTokens()];
		int count = 0;
		while (tk.hasMoreTokens()) {
			exts[count] = tk.nextToken();
			count++;
		}
		return exts;
	}

	/**
	 * Returns the defaultProperties.
	 * @return Properties
	 */
	public Properties getDefaultProperties() {
		return defaultProperties_;
	}

	/**
	 * Returns the description.
	 * @return String
	 */
	public String getDescription() {
		return description_;
	}

	/**
	 * Returns the fileExtensions.
	 * @return String[]
	 */
	public String[] getFileExtensions() {
		return fileExtensions_;
	}

	/**
	 * Returns the name.
	 * @return String
	 */
	public String getName() {
		return name_;
	}

	/**
	 * Returns the parentContentType.
	 * @return IContentType
	 */
	public IContentType getParentContentType() {
		return parentContentType_;
	}

	/**
	 * Sets the defaultProperties.
	 * @param defaultProperties The defaultProperties to set
	 */
	public void setDefaultProperties(Properties defaultProperties) {
		defaultProperties_ = defaultProperties;
	}

	/**
	 * Sets the description.
	 * @param description The description to set
	 */
	public void setDescription(String description) {
		description_ = description;
	}

	/**
	 * Sets the fileExtensions.
	 * @param fileExtensions The fileExtensions to set
	 */
	public void setFileExtensions(String[] fileExtensions) {
		fileExtensions_ = fileExtensions;
	}

	/**
	 * Sets the name.
	 * @param name The name to set
	 */
	public void setName(String name) {
		name_ = name;
	}

	/**
	 * Sets the parentContentType.
	 * @param parentContentType The parentContentType to set
	 */
	public void setParentContentType(IContentType parentContentType) {
		parentContentType_ = parentContentType;
	}

	public static IContentType getContentType(
		IContentType parent,
		String name,
		String dname,
		String filexts,
		String desc) {
		IContentType ct = getContentType(name);
		if (ct == null) {
			ct = createContentType(parent,name,dname,filexts,desc);
		}
		return ct;
	}

	public static IContentType getContentType(String name) {
		return (IContentType) contentTypes_.get(name);
	}

	private static ContentType createContentType(IContentType parent, String name, String dname,String fexts, String desc){
		ContentType ct = new ContentType(parent, name,dname, fexts, desc);
		contentTypes_.put(name, ct);
		return ct;
	}


	public static IContentType getContentType(Configuration conf) {
		IContentType type = null;
		try {
			String name = conf.getAttribute(NAME_ATTRIBUTE);
			String dname = conf.getAttribute(DISPLAY_NAME_ATTRIBUTE,name);
			String fexts = conf.getAttribute(FILE_EXTENSIONS_ATTRIBUTE);
			String desc = conf.getAttribute(DESCRIPTION_ATTRIBUTE, "");
			String parent = conf.getAttribute(PARENT_ATTRIBUTE, "none");

			type = getContentType(name);
			if (type != null) {
				return type;
			}

			IContentType parentc = null;
			if (!parent.equals("none")) {
				parentc = getContentType(parent);
			}
			
			ContentType ctype = createContentType(parentc,name,dname,fexts,desc);
			
			logger_.debug("Initializing content type: "+name);
			
			Configuration[] props = conf.getChildren();
			
			for(int i = 0; i < props.length; i++){
				String pname = props[i].getAttribute(PROPERTY_NAME_ATTRIBUTE,"");
				String description = props[i].getAttribute(DESCRIPTION_ATTRIBUTE,"");
				String pvalue = props[i].getValue("");
				ctype.setDefaultProperty(pname,pvalue);
				//ctype.setDefaultProperty(DESCRIPTION_PREFIX+pname,description);
			}
			
			type = ctype;
		} catch (Exception e) {
			e.printStackTrace();
			System.out.println("unable to load content type");
		}

		return type;
	}

	public static Enumeration getContentTypeNames(){
		return contentTypes_.keys();
	}
	
	public static Enumeration getContentTypes(){
		return contentTypes_.elements();
	}


	public void setDefaultProperty(String name, String value) {
		defaultProperties_.setProperty(name, value);
	}

	public String getDefaultProperty(String name) {
		return defaultProperties_.getProperty(name);
	}
	
	public String[] getAllowedValuesForProperty(String name){
		String val = getDefaultProperty(ALLOWED_VALUES_PREFIX+name);
		if(val == null){
			return null;
		}
		String[] allowed = parseFileExtensionString(val);
		return allowed;
	}
	

	/**
	 * Returns the displayName.
	 * @return String
	 */
	public String getDisplayName() {
		return displayName_;
	}

	/**
	 * Sets the displayName.
	 * @param displayName The displayName to set
	 */
	public void setDisplayName(String displayName) {
		displayName_ = displayName;
	}

}
